; Ultrasonic cleaner
; Y2020 Silicon Chip

	ERRORLEVEL -302
	ERRORLEVEL -306

	list P=16F1459
	#include p16f1459.inc

;Program Configuration Register 1
		__CONFIG    _CONFIG1, _FOSC_INTOSC & _WDTE_OFF & _PWRTE_ON & _MCLRE_ON & _CP_OFF & _BOREN_OFF & _CLKOUTEN_OFF &_IESO_OFF & _FCMEN_OFF

;Program Configuration Register 2
		__CONFIG    _CONFIG2, _WRT_OFF & _CPUDIV_NOCLKDIV & _USBLSCLK_24MHz & _PLLMULT_4x & _PLLEN_DISABLED & _STVREN_ON & _BORV_HI & _LPBOR_OFF & _LVP_OFF 


; Define variables at memory locations

; Bank 0 RAM

VALUE_1		equ	H'20'	; delay values 1,2,3
VALUE_2		equ	H'21'
VALUE_3		equ	H'22'
STORE3			equ	H'23'	; A/D store for aquisition
RUN			equ	H'24'	; run mode when bit 0 is set
CURRENT		equ	H'25'	; store current
TIMEOUT1		equ	H'26'	; timeout ms byte
TIMECNT1		equ	H'27'	; timeout counter ms byte
TIMECNT2		equ	H'28'	; timeout counter ls byte
DIRECTION		equ	H'29'	; increasing or decreasing for POWER
PR2_VAL		equ	H'2A'	; storage of PR2
FIRST			equ	H'2B'	; first run to set PR2 in calibration
CYCLE			equ	H'2C'	; frequency end to end cycles counter
TIMES2			equ	H'2D'	; timeout times 2
OSC_TUNE		equ	H'2E'	; OSCTUNE value, but as a counter 0-FF
OFF_FLAG		equ	H'2F'	; flag to indicate a momentary switch off
OVERC			equ	H'30'	; overcurrent counter
TUNESTO		equ	H'31'	; OSC_TUNE store for warbling	
DN_UP			equ	H'32'	; osctune down or up direction flag

; math routines
TEMP1			equ	H'68'
REMB0			equ	H'69'
TEMP2			equ	H'6A'
AARGB1      		equ H'6B'
AARGB0      		equ H'6C'	; most significant byte of argument A
BARGB1      		equ H'6D'
BARGB0      		equ H'6E'	; most significant byte of argument B
LOOPCOUNT   	equ H'6F'  	; division counter

; All Banks RAM
TEMPD1		equ	H'70'	; temp A/D store
TEMPS			equ	H'71'	; temporary working
PWR_UP_DN	equ	H'72'	; power up or down direction
ERR_FLAG		equ	H'73'	; error flags for different errors shown in code
READADDMS	equ	H'74'	; flash memory read address ms byte making up a 14 bit byte (ms byte has 6 bits)
READADDLS	equ	H'75'  	; flash memory read address ls byte (ls byte has 8 bits)
DATA_ADD		equ	H'76'	; data address
POWER			equ	H'77'	; power level setting
PWR_COMP		equ	H'78'	; compensated power value (actually a current reading assuming 12V supply) against supply voltage

; Reserved space for DATA. Data used in the ls bytes only. Written to program memory starting at H0700 
OSC1			equ	H'7A'	; osctune ms byte (used as a diagnostic only)
OSC2			equ	H'7B'	; osctune ls byte
PR1				equ	H'7C'	; PR2, ms byte 
PR2x			equ	H'7D'	; PR2 value ls byte
POWER1		equ	H'7E'	; ms byte power level
POWER2		equ	H'7F'	; Power level ls byte

; initial values

	org	H'700'		; start address of where data memory is stored for power setting
	DA				H'000' ; OSCTUNE (diagnostics only)
	DA				H'041' ; PR2 value for 45.45kHz (D65)
	DA				H'077' ; D119 for mid power
 	
; define reset and interrupt vector start addresses

	org		0  				; start at address 0000h
	goto	MAIN
	org     	4				; interrupt vector 0004h, start interrupt routine here


;***************************************************************************************
INTERRUPT
; run timer at Timer1 overflow rate 42.9ms
	movlb	D'0'
	bcf		PIR1,TMR1IF		; timer 1 overflow flag cleared

; stop timer and load D3,738 to timer 1 so it counts D61,798 rather than 65,536 if using full timer length.
; that way we have a 90minutes timer maximum (Note: see times 2 below for counter). Timer is 21s minimum
	bcf		T1CON,TMR1ON	; stopped
	movlw	H'E'				; D3738 = HE9A, use E9C to compensate for timer off 
	movwf	TMR1H			; ms byte
	movlw	H'9C'			; compensated value for timer off
	movwf	TMR1L			; ls byte
	nop						; used to get timer to accuracy although dependent on OSCTUNE
	nop
	bsf		T1CON,TMR1ON	; timer started

; rate reduced by 2
	incf		TIMES2,f		; period x 2
	btfss	TIMES2,0
	retfie					; alternate for a times 2 period

; if Timeout1 is 255 stop increasing
	movf	TIMECNT1,w
	xorlw	D'255'
	btfsc	STATUS,Z		
	retfie

; increase ls byte, increase ms byte when overflows 
	incfsz	TIMECNT2,f		; ls byte
	goto	FLAG
	incf		TIMECNT1,w		; ms byte
	btfsc	STATUS,Z
; if zero do not decrease
	retfie
	incf		TIMECNT1,f
	retfie

FLAG
	movf	TIMECNT2,w
	xorlw	D'127'
	btfsc	STATUS,Z
	bsf		OFF_FLAG,0		; set flag each 10s or so
	movf	TIMECNT2,w
	xorlw	D'1'
	btfsc	STATUS,Z
	bsf		OFF_FLAG,0		; set flag each 10s
	retfie
		
;..................................................................................................................................
MAIN

; initial values
	clrf		RUN			; run mode off
	clrf		TIMEOUT1		; timeout ms byte
	clrf		DIRECTION		; increasing or decreasing power 
	clrf		PWR_UP_DN	; power up or down direction
	clrf		ERR_FLAG		; error flag
	clrf		OSC_TUNE		; OSCTUNE value
	clrf		POWER1		; ms byte of POWER for storage
	clrf		OSC1			; ms byte for OSC_TUNE (or OSCTUNE) for diagnostic storage
	clrf		PR1				; ms byte of PR2 for PWM frequency storage 

; CWG
	movlb	D'13'			; bank 13, CWG
	bcf		CWG1CON0,G1EN	; CWG disabled initially

; set inputs/outputs
	movlb	D'2'			; latch, bank 2

	movlw	B'00000000'	; 
	movwf	LATA
	movlw	B'11010000'	; LEDs off
	movwf	LATB
	movlw	B'00000101'	; LEDs off, Q5 off, Q1,Q2 off, Power LED (LED1) on
	movwf	LATC

;USB
	movlb	D'29'
	bcf		UCON,3		; USB off
	bcf		UCON,1
	
; weak pullups off/on
	movlb	D'4'			; bank4	WPUA/B
	clrf		WPUA
	clrf		WPUB

; set I/O
	movlb	D'1'			; bank1	TRISA/B/C
	movlw   B'00001011'	; I/O 
	movwf   TRISA		; port A data direction register
	movlw	B'00100000'	; I/O (RB outputs)
	movwf	TRISB		; port B data direction register
	movlw	B'11000000'
	movwf	TRISC		; port C data direction RC4 and RC5 initially low outputs for Q1,Q2 gates off.
; options
	movlw	B'00000000'	; weak pullups set via WPUA/B
	movwf	OPTION_REG	; 
; analog inputs
	movlb	D'3'			; bank3	ANSELA/B/C
	movlw	B'00000000'	; 
	movwf	ANSELA
	movlw	B'00100000' ; AN11
	movwf	ANSELB
	movlw	B'11000000'	; AN8,AN9
	movwf	ANSELC
; A/D select
	movlb	D'1'			; bank1	ADCON0/1/2
	movlw	B'00100000'	; channel 8
; 00100000'	; channel 8
; 00101100 	; channel 11
; 00100100	; channel 9
	movwf	ADCON0
	movlw	B'01100000'	; left justified A/D result, fosc /64, Vdd to Vss A/D
	movwf	ADCON1
	bsf		ADCON0,ADON	; A/D on

; oscillator	
	movlw	B'11111100'	; for 48MHz; 16MHz x 3PLL
	movwf	OSCCON	; osc

; timer2 set
	movlb	D'0'			; bank0	
	movlw	D'69'		; PWM  at 43kHz, initial setting,
	movwf	PR2			; PWM period register

; use prescaler of /4 and 48MHz clock 
; frequency = 1/ (333.333ns x (PR2 +1))
; PR2=82 for 36kHz
; PR2=78 for 38kHz
; PR2=74 for 40kHz
; PR2=69 for 43kHz

	movlw	B'00000101'
	movwf	T2CON

; timer1 set
	movlw	B'00110001'		; 48MHz/4/8;  period 666.666ns. 
	movwf	T1CON

; PWM set
	movlb	D'12'			; bank12  PWM	
	movlw	B'11100000'		; set PWM mode
	movwf	PWM1CON		; enable PWM1 operation
	clrf		PWM1DCL
	clrf		PWM1DCH		; 0 duty cycle (use PR2/2  for ~ 50% duty cycle) 
; wait
	movlb	D'0'				; bank 0'
	call		DELAY			; 100ms and time for RC5 and RC4 to have gate of Q1 and Q2 low

; CWG. Set up complementary waveform generator (CWG) for Q1,Q2 push pull drive of transformer
	movlb	D'1'				; bank1	TRISC
	movlw	B'11110000'
	movwf	TRISC			; RC4 and RC5 inputs
	movlb	D'13'			; bank 13, CWG
; dead band
	movlw	D'31'			; using 16MHz:  62.5ns dead time /count (D63 is maximum value before rollover to 0) 31 gives 1.93us
	movwf	CWG1DBR		; rising
	movwf	CWG1DBF		; falling
; shutdown
	clrf		CWG1CON2		
	movlw	B'10100010'		; outputs at 0 in shutdown, PWM1 source
	movwf	CWG1CON1
	bsf		CWG1CON2,G1ASE	; auto shutdown status bit
	bcf		CWG1CON2,G1ARSEN; auto restart off
; outputs
	movlw	B'01100001'		; CWG1A and B enabled as complementary outputs, 16MHz dead band control
	movwf	CWG1CON0
	bsf		CWG1CON0,G1EN	; enable CWG
; outputs
	movlb	D'1'				; bank1	TRISC
	movlw	B'11000000'		; RC4,5 outputs
	movwf	TRISC			; RC4 and RC5 outputs
;	movlb	D'0'				; bank 0

; wait for PLL stable
;	movlb	D'01'				; bank 1
HF1
	btfss	OSCSTAT,PLLRDY	; phase lock loop ready check
	goto	HF1

; ensure the USB is off as it affects the RA0 and RA1 inputs for switches
	movlb	D'29'
	bcf		UCON,3			; usb off
	movlb	D'0'				; bank 0

;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
; Diagnostic TEST
; check if both switches pressed at power up
; if so run diagnostics

	btfsc	PORTA,0		; check start switch 
	goto	ALLOW_INTER	; bypasses test
	btfsc	PORTA,1		; check Stop switch	
	goto	ALLOW_INTER	; bypasses test	
; both low

; wait for both switches high (open)
WT	call		DELAY
	btfss	PORTA,0
	goto	WT	
	btfss	PORTA,1
	goto	WT

	movlb	D'2'
; all LEDs on		
	bcf		LATC,0			; LED2 on
	bcf		LATC,2			; LED3 on
	bcf		LATB,4			; LED4 on
	bcf		LATB,6			; LED5 on
	bcf		LATB,7			; LED6 on
	movlb	D'0'

; set PR2 for PWM frequency when OSCTUNE is zero
; use prescaler of /4 and 48MHz clock 
; frequency = 1/ (333.333ns x (PR2 +1))
; eg
; PR2=82 for 36kHz
; PR2=78 for 38kHz
; PR2=74 for 40kHz
; PR2=69 for 43kHz

	movlw	D'74'	; 40kHz
CHANGED
	movwf	PR2
	call		C50		; calculate 50%
	goto	PWR_SUPP

;...................................................................
C50
; calculate 50% duty in PWM
	incf		PR2,w			; increment to get actual value used for frequency
	movwf	TEMPS
	lsrf		TEMPS,w		; (PR2+1)/2 to get the 50% duty cycle value from PR2 (setting frequency). carry bit to lsb
	movlb	D'12'			; bank12  PWM	
	movwf	PWM1DCH		; duty cycle (use PR2/2  for ~ 50% duty cycle) 
	btfss	STATUS,C
	goto	CLR7c
	clrf		PWM1DCL
	bsf		PWM1DCL,7
	goto	LSB_DONEc
CLR7c
	clrf		PWM1DCL
LSB_DONEc
	movlb	D'0'
	bcf		PIR1,TMR2IF		; timer overflow flag cleared
; wait for a set timer 2 flag
WAIT_SETc
	btfss	PIR1,TMR2IF		; ensures start on complete PWM
	goto	WAIT_SETc	
	bcf		PIR1,TMR1IF	
	return
;...............................................................

PWR_SUPP
	movlb	D'2'				; bank 2 Latch
	bcf		LATC,1			; power LED on
; power up supply to transformer
	bsf		LATC,3			; Q5 on for Q6 on to switch on power
	movlb	D'0'				; bank 0

; wait for 1s
	movlw	D'250'			; 0.25s
	call		DELX
	movlw	D'250'
	call		DELX
	movlw	D'250'			; 0.25s
	call		DELX
	movlw	D'250'
	call		DELX

RVR1
; check stop switch
	btfss	PORTA,1		; check Stop switch	
	goto	OFFx

;  Drive on
	movlb	D'13'			; bank 13, CWG			
	bcf		CWG1CON2,G1ASE	; auto shutdown status bit, set to stop CWG, clear to start
	movlb	D'0'

; prevent over current
; read current AN11
	movlb	D'1'				; bank1	ADCON0/1/2
	movlw	B'00101100' 		; channel 11
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	movlb	D'0'				; bank 0
	call		ACQUIRE_AD	; result in TEMPD1
	movf	TEMPD1,w
	movwf	CURRENT		; store value for comparison

; if over 250 then overcurrent so error (equivalent to 4.9V and when divided by amplifier gain of 28 gives 171.56mV. Current is 2 x 171.56/0.1) = 3.5A 
	movlw	D'250'
	subwf	CURRENT,w
	btfss	STATUS,C		; if C is positive (set then over)
	goto	UNDER_I

; switch off drive
	movlb	D'13'			; bank 13, CWG			
	bsf		CWG1CON2,G1ASE	; auto shutdown status bit, set to stop CWG
	
	movlb	D'2'
; odd numbered LEDs off
	bcf		LATC,0			; LED2 on
	bsf		LATC,2			; LED3 off
	bcf		LATB,4			; LED4 on
	bsf		LATB,6			; LED5 off
	bcf		LATB,7			; LED6 on	
	movlb	D'0'				; bank 0
	
	movlw	D'250'			; 0.25s
	call		DELX
	movlw	D'250'			; 0.25s
	call		DELX

	goto	READ_TEST

UNDER_I

; all LEDs on
	movlb	D'2'
	bcf		LATC,0			; LED2 on
	bcf		LATC,2			; LED3 on
	bcf		LATB,4			; LED4 on
	bcf		LATB,6			; LED5 on
	bcf		LATB,7			; LED6 on	
	movlb	D'0'				; bank 0

READ_TEST
; Read VR1 timer value AN9
	movlb	D'1'				; bank1	ADCON0/1/2
	movlw	B'00100100'		; channel 9
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	movlb	D'0'				; bank 0
	call		ACQUIRE_AD	; result in TEMPD1
	comf	TEMPD1,w
	movwf	OSC_TUNE
	call		CONVERT		; convert for 2'complement OSCTUNE value

; if OSC_TUNE is 0 or FF, increase or decrease PR2 to limits or D65 (45.45kHz) to D85 (34.88kHz)
	movf	OSC_TUNE,w
	btfsc	STATUS,Z
	goto	LESS_PR2
	xorlw	H'FF'
	btfss	STATUS,Z
	goto	RVR1

MORE_PR2
; OSC_TUNE needs to be at 0 and Start switch pressed before any change to PR2
	btfsc	PORTA,0
	goto	RVR1

	incf		PR2,w
	sublw	D'85'
	btfss	STATUS,C		; stop at 85
	goto	BY1				; no change to PR2
	incf		PR2,f
	call		C50				; 50% duty calculation
BY1	call		DELAY
; wait for switch open	
	btfss	PORTA,0
	goto	BY1
	goto	RVR1

LESS_PR2
; OSC_TUNE needs to be at 0 for period before any change to PR2
	btfsc	PORTA,0
	goto	RVR1

	decf	PR2,w
	sublw	D'65'
	btfsc	STATUS,C
	goto	BY1				; no change to PR2
	decf	PR2,f
	call		C50				; 50% duty calculation
	goto	BY1

OFFx; resume normal program set to off
	movlb	D'1'
	bsf		PIE1,TMR1IE		; enable timer1 interrupt
	movlb	D'0'
	bcf		PIR1,TMR1IF		; flag clear
	bsf		INTCON,PEIE	; enable peripheral interrupts
	goto	OFF

; end test	
;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

ALLOW_INTER
; interrupts
	movlb	D'1'
	bsf		PIE1,TMR1IE		; enable timer1 interrupt
	movlb	D'0'
	bcf		PIR1,TMR1IF		; flag clear
	bsf		INTCON,PEIE	; enable peripheral interrupts

;............................................................................................................................................................................

; check switches for on/off
SWITCHES
; check if in run mode so only check for off
	btfss	RUN,0
	goto	SWITCHES_ALL
	btfss	PORTA,1		; check Stop switch	
	goto	OFF

	btfsc	PORTA,0		; if run switch pressed, adjust power
	goto	LOOP1			; drive ultrasonic Run switch is off
; PWM off
	movlb	D'13'			; bank 13, CWG			
	bsf		CWG1CON2,G1ASE	; auto shutdown status bit, set to stop CWG
	movlb	D'0'				; bank 0
	goto	POWER_SET	; adjust power


SWITCHES_ALL
	btfss	PORTA,0		; check start switch 
	goto	ON
	btfsc	PORTA,1		; check Stop switch	
	goto	SWITCHES		; no switch is closed

;............................................................................................................................................

OFF ; STOP ; off switch pressed or timeout

; if power is already off via PORTC,3 then bypass shutdown
	btfss	PORTC,3
	goto	WAIT_STOP ; wait for stop switch off
OFF1; from error
	bcf		INTCON,GIE		; interrupts off
	clrf		RUN			; run mode off

	movlb	D'2'				; bank 2 Latch
; power off supply to transformer
	bcf		LATC,3			; Q5 off for Q6 on to switch off power
; LEDs 2-6 off
	bsf		LATC,0			; LED2 off
	bsf		LATC,2			; LED3 off
	bsf		LATB,4			; LED4 off
	bsf		LATB,6			; LED5 off
	bsf		LATB,7			; LED6 off
; LED1 on
	bcf		LATC,1			; Power LED on	

	movlb	D'13'			; bank 13, CWG			
	bsf		CWG1CON2,G1ASE	; auto shutdown status bit, set to stop CWG
	movlb	D'0'				; bank 0


WAIT_STOP ; wait for stop switch off
	call		DELAY
	btfss	PORTA,1		; Stop switch
	goto	WAIT_STOP1	; wait until open
	call		DELAY			; wait and check again if open
	btfss	PORTA,1		; Stop switch
	goto	WAIT_STOP		; wait until open
	goto	SWITCHES	
WAIT_STOP1

; if both switches, run calibrate
; check on switch
	btfsc	PORTA,0		; if on, run calibrate
	goto	WAIT_STOP	

; both switches are on
; show all LEDs

	movlb	D'2'
; all LEDs on		
	bcf		LATC,0			; LED2 on
	bcf		LATC,2			; LED3 on
	bcf		LATB,4			; LED4 on
	bcf		LATB,6			; LED5 on
	bcf		LATB,7			; LED6 on
	movlb	D'0'

; wait for both switches off
WAIT_OP ; wait for switches off
	call		DELAY
	btfss	PORTA,1		; Stop switch
	goto	WAIT_OP		; wait until open
	call		DELAY			; wait and check again if open
	btfss	PORTA,0		; ON/run switch
	goto	WAIT_OP		; wait until open
	
	movlb	D'2'
; all LEDs off		
	bsf		LATC,0			; LED2 off
	bsf		LATC,2			; LED3 off
	bsf		LATB,4			; LED4 off
	bsf		LATB,6			; LED5 off
	bsf		LATB,7			; LED6 off
	movlb	D'0'

	clrf		FIRST			; first run to adjust PR2
	movlw	H'7F'
	movwf	OSC_TUNE		; OSC_TUNE value centred
	movlb	D'1'
	clrf		OSCTUNE		; OSCTUNE at default centre of frequency adjustment
	movlb	D'0'
	goto	ON1
;....................................................................................................................................................................................

ON ; START
;...........................................................................................................................................................................
; Automatic Calibration if PR2=65 then run calibration
; Read stored value
	movlw	H'07'			; ms address byte (H0701)
	movwf	READADDMS 
	movlw	H'01'
	movwf	READADDLS	; ls address byte
	call		READ			; 'w' has data
	xorlw	D'65'			; original preloaded value
	btfss	STATUS,Z		; if zero run calibrate
	goto	SET_FIRST		; bypass calibrate

	clrf		FIRST			; first run to calibrate and adjust PR2
	movlw	H'7F'
	movwf	OSC_TUNE		; OSC_TUNE value centred
	movlb	D'1'
	clrf		OSCTUNE		; OSCTUNE at default centre of frequency adjustment
	movlb	D'0'
	goto	ON1
;...................................................

SET_FIRST

	bsf		FIRST,0			; only cleared for when running calibration
; load osctune
	movlw	H'00'
	movwf	OSC_TUNE		; OSC_TUNE value 
	movlb	D'1'
	movlw	H'3F'
	movwf	OSCTUNE		; OSCTUNE at maximum frequency
	movlb	D'0'
	
ON1
	clrf		OVERC			; overcurrent counter

	bsf		RUN,0			; run mode started

	movlb	D'2'				; bank 2 Latch
	bcf		LATC,1			; power LED on
; power up supply to transformer
	bsf		LATC,3			; Q5 on for Q6 on to switch on power
	movlb	D'0'				; bank 0
; wait for 0.5s
	movlw	D'250'			; 0.25s
	call		DELX
	movlw	D'250'
	call		DELX

	movlb	D'2'				; bank 2 Latch
	bcf		LATC,3			; Q5 off for Q6 off to switch off power
	movlb	D'0'				; bank 0

	movlw	D'250'
	call		DELX
;  check for voltage rise at AN8
	movlb	D'1'			; bank1	ADCON0/1/2
	movlw	B'00100000'	; channel 8
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	movlb	D'0'			; bank 0
	call		ACQUIRE_AD	; result in TEMPD1

; initially set ERR_FLAG of error display code
	clrf		ERR_FLAG

; error if voltage does not rise
; should be overall 12V or 4V at AN8 input  (D204 via A/D)
; check if voltage is over about 9V or 3V at AN8 (D152 at A/D)
	movlw	D'152'
	subwf	TEMPD1,w
	btfss	STATUS,C		; if negative, (C=0) then an error 
	goto	ERRORX		; 

; .........................................................................................................................................

POWER_SET
; power up supply to transformer
	movlb	D'2'				; bank 2 Latch
	bsf		LATC,3			; Q5 on for Q6 on to switch on power
	movlb	D'0'				; bank 0

; bypass power adjust when in start mode (calibrate)
	movf	FIRST,w
	btfsc	STATUS,Z
	goto	READ_VR1		; bypass

; adjust power level
; is Start switch closed
	btfsc	PORTA,0
	goto	READ_VR1		

; Read stored POWER value
	movlw	H'07'			; ms address byte (H0702 )
	movwf	READADDMS 
	movlw	H'02'
	movwf	READADDLS	; ls address byte
	call		READ			; 'w' has data
	movwf	POWER
	call		PWR_DSP		; power display

; change power from between 0 and 200
; check direction
	btfsc	PWR_UP_DN,0	; power up or down direction
	goto	LOOP_DN

LOOP_UP
	clrf		PWR_UP_DN	; power up or down direction. set for up

	movf	POWER,w		; no increase if at maximum 
	sublw	D'213'
	btfss	STATUS,C
	goto	LOOP_DN		; reverse direction
	movlw	D'24'
	addwf	POWER,w
	movwf	TEMPS
	sublw	D'214'
	movf	TEMPS,w			; ready added value
	btfss	STATUS,C
	movlw	D'214'
	movwf	POWER
	call		PWR_DSP		; power display

; delay
	movlw	D'255'
	call		DELX
	movlw	D'255'
	call		DELX
	btfsc	PORTA,0		; check switch
	goto	STORE_PWR
	goto	LOOP_UP

LOOP_DN
	bsf		PWR_UP_DN,0	; power up or down direction, set for down

	movf	POWER,w		; no decrement if zero
	btfsc	STATUS,Z
	goto	LOOP_UP
	movlw	D'24'
	subwf	POWER,w
	btfss	STATUS,C		; if negative set at 0
	movlw	D'0'				; minimum power
	movwf	POWER
	call		PWR_DSP		; power display

; delay
	movlw	D'255'
	call		DELX
	movlw	D'255'
	call		DELX
	btfss	PORTA,0		; check switch
	goto	LOOP_DN

STORE_PWR; store power 
	call		PWR_DSP		; power display

;........................................................................................................................
; optional, reverse POWER direction each time switch pressed. 
; !!!!!!!!!!!!!!!
	goto	LOAD			; Remark in or out to include or bypass

; swap direction 
	btfss	PWR_UP_DN,0
	goto	SET_UP_DN
	clrf		PWR_UP_DN
	goto	LOAD
SET_UP_DN	
	bsf		PWR_UP_DN,0
; end of optional reverse
;..........................................................................................................................

LOAD; Write POWER value 
;Write  values 

	movlb	D'0'
	movf	OSC_TUNE,w
	movlb	D'0'
	movwf	OSC2
	movf	PR2_VAL,w
	movwf	PR2x	
	movf	POWER,w
	movwf	POWER2
	call		WRITE1			; erase and write to flash memory

;.............................................................................................................................................

READ_VR1
; Read VR1 timer value AN9
	movlb	D'1'			; bank1	ADCON0/1/2
	movlw	B'00100100'	; channel 9
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	movlb	D'0'			; bank 0
	call		ACQUIRE_AD	; result in TEMPD1
;  place A/D value in ms byte TIMEOUT1). Minimum is 256 x 41.2ms = 21s. Maximum is 256 x 256 (or 65,536) x  41.2ms or 90minutes

	movf	TEMPD1,w
	movwf	TIMEOUT1
; check if zero
	movf	TIMEOUT1,w
	btfsc	STATUS,Z
	incf		TIMEOUT1,f		; minimum of 1
; clear counters for timer used in interrupt
	clrf		TIMECNT1
	clrf		TIMECNT2

	bsf		INTCON,GIE		; start timer

; Read stored POWER value
	movlw	H'07'			; ms address byte (H0702)
	movwf	READADDMS 
	movlw	H'02'
	movwf	READADDLS	; ls address byte
	call		READ			; 'w' has data
	movwf	POWER

; Read stored PR2 (PWM frequency)
; set up frequency
	clrf		DIRECTION		; 
	movlw	H'07'			; ms address byte (H0701)
	movwf	READADDMS 
	movlw	H'01'
	movwf	READADDLS	; ls address byte
	call		READ			; 'w' has data
	movwf	PR2				; PWM period register
	movwf	PR2_VAL		; for use in interrupt

; calculate 50% duty in PWM
	incf		PR2,w			; increment to get actual value used for frequency
	movwf	TEMPS
	lsrf		TEMPS,w		; (PR2+1)/2 to get the 50% duty cycle value from PR2 (setting frequency). carry bit to lsb
	movlb	D'12'			; bank12  PWM	
	movwf	PWM1DCH		; duty cycle (use PR2/2  for ~ 50% duty cycle) 
	btfss	STATUS,C
	goto	CLR7
	clrf		PWM1DCL
	bsf		PWM1DCL,7
	goto	LSB_DONE
CLR7
	clrf		PWM1DCL
LSB_DONE
	movlb	D'0'
	bcf		PIR1,TMR2IF		; timer overflow flag cleared
; wait for a set timer 2 flag
WAIT_SET
	btfss	PIR1,TMR2IF		; ensures start on complete PWM
	goto	WAIT_SET	
	bcf		PIR1,TMR1IF	

	movlb	D'13'			; bank 13, CWG			
	bcf		CWG1CON2,G1ASE	; auto shutdown status bit, clear to start CWG
	movlb	D'0'				; bank 0

;  power display

	call		PWR_DSP		; power display routine
	
	movlb	D'2'
	bsf		LATC,1			; Power LED off
	movlb	D'0'

	clrf		CYCLE			; out of range counter for power counter

	call		POWER_CALC	; compensate power according to supply voltage

LOOP1; goes to here from 'switches' when no switch is detected as closed during running (ie when driving transducer)

; Read VR1 timer value AN9
	movlb	D'1'			; bank1	ADCON0/1/2
	movlw	B'00100100'	; channel 9
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	movlb	D'0'			; bank 0
	call		ACQUIRE_AD	; result in TEMPD1
;  place A/D value in ms byte TIMEOUT1). Minimum is 256 x 41.2ms = 21s. Maximum is 256 x 256 (or 65,536) x  41.2ms or 90minutes

	movf	TEMPD1,w
	movwf	TIMEOUT1
; check if zero
	movf	TIMEOUT1,w
	btfsc	STATUS,Z
	incf		TIMEOUT1,f		; minimum of 1

; check timer for timeout. TIMECNT1 is = or > than TIMEOUT1
	movf	TIMEOUT1,w
	subwf	TIMECNT1,w
	btfsc	STATUS,C		
	goto	OFF			; timeout so end

	movlw	D'10'			; xms delay
	call		DELX			; 1ms per value

; read current AN11 (error 2 if over)
	movlb	D'1'				; bank1	ADCON0/1/2
	movlw	B'00101100' 		; channel 11
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	movlb	D'0'				; bank 0
	call		ACQUIRE_AD	; result in TEMPD1
	movf	TEMPD1,w
	movwf	CURRENT		; store value for comparison

OVER_CK
; if over 250 then overcurrent so error (equivalent to 4.9V and when divided by amplifier gain of 28 gives 171.56mV. Current is 2 x 171.56/0.1) = 3.5A 
	movlw	D'250'
	subwf	CURRENT,w
	btfss	STATUS,C		; if C is positive (set then over)
	goto	ADJUST_POWER

; count overcurrent over 8 counts
	bsf		STATUS,C
	rlf		OVERC,f		; overcurrent counter roll carry
; if all 1's then overcurrent occurred 8 times in a row so overload 
	incfsz	OVERC,w		; if FF then an increment will overflow to zero
	goto	ADJUST_POWER
; prepare for error
	movlb	D'13'			; bank 13, CWG			
	bsf		CWG1CON2,G1ASE	; auto shutdown status bit, set to stop CWG
	movlb	D'0'				; bank 0
	bsf		ERR_FLAG,1	; not zero so shows overload error
	goto	ERRORX

ADJUST_POWER ; adjust power to setting 

; wait until power LED is on (locked in frequency loop) before momentary frequency changes
	btfsc	PORTC,1
	goto	COMPARE

; check off flag
	btfss	OFF_FLAG,0		; if set then stop output momentarily (set in interrupt)
	goto	COMPARE

; non control period and shift to higher frequency and return
	clrf		OFF_FLAG

; Power LED off
	movlb	D'2'
	bsf		LATC,1			; power LED off
	movlb	D'0'

; store OSC_TUNE then reduce OSCTUNE to 0 (increasing frequency) and back to original as a frequency variation for standing wave control
	movf	OSC_TUNE,w
	movwf	TUNESTO
	btfsc	STATUS,Z	; if zero then bypass
	goto	P_CALC
	bcf		DN_UP,0	; start as down

CYC_DN_UP	
; check if down or up
	btfsc	DN_UP,0
	goto	UPx
; decrease OSC_TUNE to zero
	decfsz	OSC_TUNE,f
	goto	DNx
; reverse to increase 
	bsf		DN_UP,0
	goto	DNx
UPx
; increase until = TUNESTO	
	incf		OSC_TUNE,w
	xorwf	TUNESTO,w
	btfsc	STATUS,Z		; when equal exit
	goto	COMPARE
	incf		OSC_TUNE,f
DNx
	call		CONVERT		; converts for OSCTUNE value

; delay for off period
	movlw	D'1'			; 2ms
	call		DELX
	goto	CYC_DN_UP

P_CALC
	call		POWER_CALC	; compensate power according to voltage
	goto	COMPARE

; small subroutine for above
; convert OSC_TUNE (00-FF) to OSCTUNE 7-bit 2's complement
CONVERT
	movf	OSC_TUNE,w
	movwf	TEMPS
	lsrf		TEMPS,f
	btfss	TEMPS,6
	goto	REVx
	comf	TEMPS,f
	bcf		TEMPS,7
	bsf		TEMPS,6
	goto	STRAIGHTx
REVx
	comf	TEMPS,f
	bcf		TEMPS,7
	bcf		TEMPS,6
STRAIGHTx
	movf	TEMPS,w
	movlb	D'1'
	movwf	OSCTUNE
	movlb	D'0'	
	return

;........................................................................................................................................

COMPARE; current against power

	btfss	FIRST,0			; when set then run osctune adjust
	goto	RUN_START

; First check if current is <4 
	movlw	D'4'
	subwf	CURRENT,w
	btfss	STATUS,C
	goto	CK0			; less than 4 so bypass window control

; provide window band  for current comparison with power

; 1 above and below
	movlw	D'1'
	subwf	CURRENT,w
	btfsc	STATUS,Z
	goto	PREP_SWITCH	; equal so no change
	movlw	D'1'
	addwf	CURRENT,w
	btfsc	STATUS,Z
	goto	PREP_SWITCH	; equal so no change

CK0 ; check if equal
	movf	CURRENT,w
	subwf	PWR_COMP,w
	btfsc	STATUS,Z		; when the same
	goto	PREP_SWITCH	; equal so no change
	btfss	STATUS,C
	goto	SET_TRANS1	
	incfsz	OSC_TUNE,f		; raise frequency
	goto	TRANSF
	goto	DEC
SET_TRANS1
	decfsz	OSC_TUNE,f		; decrease frequency
	goto	TRANSF
	goto	INC	
TRANSF ; convert OSC_TUNE (00-FF) to OSCTUNE 7-bit 2's complement
	movf	OSC_TUNE,w
	movwf	TEMPS
	lsrf		TEMPS,f
	btfss	TEMPS,6
	goto	REV
	comf	TEMPS,f
	bcf		TEMPS,7
	bsf		TEMPS,6
	goto	STRAIGHT
REV
	comf	TEMPS,f
	bcf		TEMPS,7
	bcf		TEMPS,6
STRAIGHT
	movf	TEMPS,w
	movlb	D'1'
	movwf	OSCTUNE
	movlb	D'0'

	goto	SWITCHES

PREP_SWITCH
; switch on power LED when in lock
; If Power LED is on bypass
	btfss	PORTC,1
	goto	SWITCHES

	movlb	D'2'
	bcf		LATC,1			; power LED on
	movlb	D'0'
	goto	SWITCHES

CYCLING
; check frequency cycling where power cannot be delivered and frequency is varied to find the transducer current
	movlb	D'2'
	bsf		LATC,1			; power LED off
	movlb	D'0'

; lower or increase POWER if reaches a count of 2 or more
	movlw	D'2'
	subwf	CYCLE,w
	btfss	STATUS,C		; if negative (C=0) ok
;
	goto	SWITCHES

	btfss	DIRECTION,0	; if clear increase power
	goto	INC_POWER
; reduce POWER value 
	clrf		CYCLE
	movf	POWER,w		; no decrement if zero
	btfsc	STATUS,Z
	goto	SWITCHES
	movlw	D'24'
	subwf	POWER,w
	btfss	STATUS,C		; if negative set at 0
	movlw	D'0'				; minimum power
	goto	LD_PWR

INC_POWER
; increase power
	clrf		CYCLE
	movf	POWER,w		; no increase if at maximum (216)
	sublw	D'216'
	btfss	STATUS,C
	goto	SWITCHES
	movlw	D'24'
	addwf	POWER,w
	movwf	TEMPS
	sublw	D'216'
	movf	TEMPS,w		; ready added value
	btfss	STATUS,C
	movlw	D'216'
LD_PWR
	movwf	POWER
	call		PWR_DSP		; show new power value
	call		POWER_CALC	; compensate power according to voltage

;Write  values to program memory

	movlb	D'0'
	movf	OSC_TUNE,w	; could use OSCTUNE at bank 1 to check value
	movlb	D'0'
	movwf	OSC2
	movf	PR2,w
	movwf	PR2x	
	movf	POWER,w
	movwf	POWER2
	call		WRITE1			; erase and write to flash memory

	bsf		 INTCON,GIE 	; re-enable interrupts 
	goto	SWITCHES

DEC
	incf		CYCLE,f			; counts frequency cycling
	bsf		DIRECTION,0	; decrease on
	goto	CYCLING
INC
	incf		CYCLE,f			; counts frequency cycling
	bcf		DIRECTION,0	; increase on
	goto	CYCLING

;..........................................................................................................................................................
; CALIBRATION

RUN_START; calibrate for PR2
; Startup to set PR2; adjust PR2 for a current reading to get frequency approaching resonance

	movf	CURRENT,w
	sublw	D'50'
	btfss	STATUS,C
	goto	SET_TRANSF11	

	incf		PR2_VAL,f		; lower frequency
; check if goes above 85 for PR2 (34.88kHz)
	movf	PR2_VAL,w
	sublw	D'85'
	btfsc	STATUS,C		
	goto	TRANSF1
	
	movlb	D'2'
	bcf		LATC,1			; Power LED on
	movlb	D'0'
	goto	OFF

SET_TRANSF11
; decrease frequency
	decf	PR2_VAL,f
; end initial PR2 setup
	bsf		FIRST,0

	movlb	D'2'
	bsf		LATC,1			; Power LED off
	movlb	D'0'

; load osctune values
	movlw	H'00'
	movwf	OSC_TUNE		; OSC_TUNE value 
	movlb	D'1'
	movlw	H'3F'
	movwf	OSCTUNE		; OSCTUNE at maximum frequency
	movlb	D'0'

;Write  values to program memory

	movlb	D'0'				; could use D'1' and OSCTUNE
	movf	OSC_TUNE,w
	movlb	D'0'
	movwf	OSC2
	movf	PR2_VAL,w
	movwf	PR2x	
	movf	POWER,w
	movwf	POWER2
	call		WRITE1			; erase and write to flash memory

	bsf		 INTCON,GIE 	; re-enable interrupts 

TRANSF1
; transfer value to PWM (PR2 and duty)
; wait for a set timer 2 flag
	bcf		PIR1,TMR2IF		; clear timer flag
WAIT_SET11
	btfss	PIR1,TMR2IF		; ensures start on complete PWM for 50% duty calculation
	goto	WAIT_SET11	
	bcf		PIR1,TMR2IF	
	movf	PR2_VAL,w		; transfer 
	movwf	PR2

; calculate 50% duty in PWM
	incf		PR2,w			; increment to get actual value used for frequency
	movwf	TEMPS
	lsrf		TEMPS,w		; (PR2+1)/2 to get the 50% duty cycle value from PR2 (setting frequency). carry bit to lsb
	movlb	D'12'			; bank12  PWM	
	movwf	PWM1DCH		; duty cycle (use PR2/2  for ~ 50% duty cycle) 
	btfss	STATUS,C
	goto	CLR_7
	clrf		PWM1DCL
	bsf		PWM1DCL,7
	goto	LSB_DONE1
CLR_7
	clrf		PWM1DCL
LSB_DONE1
	movlb	D'0'
	goto	SWITCHES

;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
; subroutines

PWR_DSP	; power display
	movlb	D'2'				; prepare for driving LEDs
	movlw	D'200'
	subwf	POWER,w
	btfsc	STATUS,C		; if positive (C=1) then LED 6
	goto	LED6

	movlw	D'176'
	subwf	POWER,w
	btfsc	STATUS,C		; if positive then LED5 and 6
	goto	LED5_6

	movlw	D'152'
	subwf	POWER,w
	btfsc	STATUS,C		; if positive then LED5 
	goto	LED5
	
	movlw	D'128'
	subwf	POWER,w
	btfsc	STATUS,C		; if positive then LED4 and 5
	goto	LED4_5

	movlw	D'104'
	subwf	POWER,w
	btfsc	STATUS,C		; if positive then LED4
	goto	LED4

	movlw	D'80'
	subwf	POWER,w
	btfsc	STATUS,C		; if positive then LED3 and 4
	goto	LED3_4

	movlw	D'56'
	subwf	POWER,w
	btfsc	STATUS,C		; if positive then LED3
	goto	LED3

	movlw	D'32'
	subwf	POWER,w
	btfsc	STATUS,C		; if positive then LED2 and 3
	goto	LED2_3

LED2	
	bcf		LATC,0			; LED2 on
	bsf		LATC,2			; LED3 
	bsf		LATB,4			; LED4 
	bsf		LATB,6			; LED5 
	bsf		LATB,7			; LED6 	
	goto	ADJ_PWR

LED2_3	
	bcf		LATC,0			; LED2 on
	bcf		LATC,2			; LED3 on
	bsf		LATB,4			; LED4 
	bsf		LATB,6			; LED5 
	bsf		LATB,7			; LED6 	
	goto	ADJ_PWR

LED3	
	bsf		LATC,0			; LED2 
	bcf		LATC,2			; LED3 on
	bsf		LATB,4			; LED4
	bsf		LATB,6			; LED5
	bsf		LATB,7			; LED6	
	goto	ADJ_PWR

LED3_4	
	bsf		LATC,0			; LED2 
	bcf		LATC,2			; LED3 on
	bcf		LATB,4			; LED4 on
	bsf		LATB,6			; LED5 
	bsf		LATB,7			; LED6 	
	goto	ADJ_PWR

LED4	
	bsf		LATC,0			; LED2 
	bsf		LATC,2			; LED3 
	bcf		LATB,4			; LED4 on
	bsf		LATB,6			; LED5 
	bsf		LATB,7			; LED6 	
	goto	ADJ_PWR

LED4_5	
	bsf		LATC,0			; LED2 
	bsf		LATC,2			; LED3 
	bcf		LATB,4			; LED4 on
	bcf		LATB,6			; LED5 on
	bsf		LATB,7			; LED6 	
	goto	ADJ_PWR

LED5	
	bsf		LATC,0			; LED2 
	bsf		LATC,2			; LED3 
	bsf		LATB,4			; LED4 
	bcf		LATB,6			; LED5 on
	bsf		LATB,7			; LED6 	
	goto	ADJ_PWR

LED5_6
	bsf		LATC,0			; LED2 
	bsf		LATC,2			; LED3 
	bsf		LATB,4			; LED4 
	bcf		LATB,6			; LED5 on
	bcf		LATB,7			; LED6 on	
	goto	ADJ_PWR

LED6	
	bsf		LATC,0			; LED2 
	bsf		LATC,2			; LED3 
	bsf		LATB,4			; LED4 
	bsf		LATB,6			; LED5 
	bcf		LATB,7			; LED6 on	

ADJ_PWR
	movlb	D'0'			; Bank0
	return
;...................................................................................................................................................................
ERRORX
; error when supply voltage does not rise
	movlb	D'2'				; bank 2 Latch

	bsf		LATC,1			; power LED off
; power off supply to transformer
	bcf		LATC,3			; Q5 off for Q6 to switch off power

FLASH_ERROR
; flash LEDs 2-6 depending on ERR_FLAG value
	movf	ERR_FLAG,w
	btfss	STATUS,Z		; when zero flash on all LEDS
	goto	ERR2
; all LEDs		
	bcf		LATC,0			; LED2 on
	bcf		LATC,2			; LED3 on
	bcf		LATB,4			; LED4 on
	bcf		LATB,6			; LED5 on
	bcf		LATB,7			; LED6 on	
	goto	WAITX
ERR2
; Even numbered LEDs
	bcf		LATC,0			; LED2 on
	bcf		LATB,4			; LED4 on
	bcf		LATB,7			; LED6 on	
	
WAITX
	movlb	D'0'				; bank 0
; wait for 0.5s

	btfss	PORTA,1		; Stop switch pressed so exit
	goto	OFF1	
	movlw	D'250'			; 0.25s
	call		DELX
	btfss	PORTA,1		; Stop switch pressed so exit
	goto	OFF1
	movlw	D'250'			; 0.25s
	call		DELX

	movlb	D'2'
; LEDs 2-6 off
	bsf		LATC,0			; LED2 off
	bsf		LATC,2			; LED3 off
	bsf		LATB,4			; LED4 off
	bsf		LATB,6			; LED5 off
	bsf		LATB,7			; LED6 off	

	movlb	D'0'				; bank 0
; wait for 0.5s
	btfss	PORTA,1		; Stop switch pressed so exit
	goto	OFF1	
	movlw	D'250'			; 0.25s
	call		DELX
	btfss	PORTA,1		; Stop switch pressed so exit
	goto	OFF1
	movlw	D'250'			; 0.25s
	call		DELX

	movlb	D'2'				; bank 2 Latch
	goto	FLASH_ERROR

;........................................................................................................
; delay using 48MHz clock
DELAY		movlw	D'100'		; 100ms 
DELX		movwf	VALUE_3	; 1ms per value
DELT_1		movlw	D'16'		; set delay period 
			movwf	VALUE_1	; VALUE_1 = w
			movlw	D'250'		; set delay period value 2 
LP_1		movwf	VALUE_2	; VALUE_2 = w
LP_2		decfsz	VALUE_2,f	; decrease VALUE_2, skip if zero
			goto 	LP_2
			decfsz	VALUE_1,f	; decrease VALUE_1, skip if zero
			goto	LP_1
			decfsz	VALUE_3,f
			goto	DELT_1	
			return	

; subroutine to wait for A/D conversion 
ACQUIRE_AD

; wait >8us to charge input capacitance 
	movlw	D'50'
	movwf	STORE3
WAIT2C1
	decfsz	STORE3,f
	goto	WAIT2C1
	movlb	D'1'			; bank 1
	bsf		ADCON0,1	; GO/DONE bit start conversion
WAIT_CONV
	btfsc	ADCON0,1	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV
	movf	ADRESH,w
	movwf	TEMPD1	; store
	movlb	D'0'			; bank0
	return

;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
; read data memory
READ; 'w' has read data
	movlb	D'3'				;  bank 3 
	movf	READADDMS,w 
	movwf 	PMADRH		; ms Byte of Program Address to read
	movf	READADDLS,w
	movwf 	PMADRL 		; ls Byte of Program Address to read
	bcf		PMCON1,CFGS	; avoid configuration space
	bsf	 	PMCON1,RD 	; Read
	nop 					
	nop
; memory is read in second cycle PM read instruction
	movf	PMDATH,w 		; ms Byte of Program data 
;	movwf	TEMP_VAL
	movf	PMDATL,w 		; ls Byte of Program data
	movlb	D'0'				; bank 0
	return
;.............................................

; write to data memory
WRITE1

; erase first before write
	bcf		 INTCON,GIE 	; Disable interrupts so required sequences will execute properly
	movlb	 D'03'
	movlw	 D'07'			 ; Load initial program memory address
	movwf	 PMADRH ;
	movlw	 D'00'
	movwf	 PMADRL ;
;
	bcf		 PMCON1,CFGS ; Not configuration space
	bsf		 PMCON1,FREE ; Enable erase
	bsf		 PMCON1,WREN ; Enable writes
	
	movlw	 H'55'		 	; Start of required write sequence:
	movwf	 PMCON2 		; Write 55h
	movlw	 H'AA' ;
	movwf	 PMCON2		 ; Write AAh
	bsf		 PMCON1,WR	 ; Set WR bit to begin write
	nop
	nop
	bcf		 PMCON1,WREN ; Disable writes

	movlw	 D'07'			 ; Load initial program memory address
	movwf	 PMADRH ;
	movlw	 D'00'
	movwf	 PMADRL ;

	bcf		 PMCON1,CFGS ; Not configuration space
	bsf		 PMCON1,WREN ; Enable writes
	bsf		 PMCON1,LWLO ; Only Load Write Latches

	movf	OSC1,w
	movwf	 PMDATH ;
	movf	OSC2,w
	movwf	 PMDATL ;

	movlw	 H'55'		 ; Start of required write sequence:
	movwf	 PMCON2 	; Write 55h
	movlw	 H'AA' ;
	movwf	 PMCON2	 ; Write AAh
	bsf		 PMCON1,WR ; Set WR bit to begin write
	nop					 ; nop instructions are forced as processor loads program memory write latches
	nop ;
	incf		 PMADRL,f	 ; Still loading latches Increment address
	
	movf	 PR1,w
	movwf	 PMDATH ;
	movf	 PR2x,w
	movwf	 PMDATL ;

	movlw	 H'55'		 ; Start of required write sequence:
	movwf	 PMCON2 	; Write 55h
	movlw	 H'AA' ;
	movwf	 PMCON2	 ; Write AAh
	bsf		 PMCON1,WR ; Set WR bit to begin write
	nop					 ; nop instructions are forced as processor loads program memory write latches
	nop ;
	incf		PMADRL,f	 ; Still loading latches Increment address

	movf	 POWER1,w
	movwf	 PMDATH ;
	movf	 POWER2,w
	movwf	 PMDATL ;

	movlw	 H'55'		 ; Start of required write sequence:
	movwf	 PMCON2 	; Write 55h
	movlw	 H'AA' ;
	movwf	 PMCON2	 ; Write AAh
	bsf		 PMCON1,WR ; Set WR bit to begin write
	nop					 ; nop instructions are forced as processor loads program memory write latches
	nop ;

START_WRITE
	bcf		PMCON1,LWLO ; No more loading latches - Start Flash program
; memory write
	movlw	H'55'		 ; Start of required write sequence:
	movwf	PMCON2 	; Write 55h
	movlw	H'AA' ;
	movwf	PMCON2 	; Write AAh
	bsf		PMCON1,WR ; Set WR bit to begin write
	nop					 ; NOP instructions are forced as processor writes all the program memory write latches simultaneously to program memory.
	nop	  
	bcf		PMCON1,WREN ; Disable writes
	movlb	 D'0'
	return

;......................................................................................................................................................................................................

POWER_CALC; compensated power against supply voltage. The power (POWER reading is actually a current reading assuming a 12V supply (reading of 204 at AN8))
; so calculate revised power against different supply voltage with the AN8 reading 

;  read voltage at AN8
	movlb	D'1'			; bank1	ADCON0/1/2
	movlw	B'00100000'	; channel 8
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	movlb	D'0'			; bank 0
	call		ACQUIRE_AD	; result in TEMPD1

; multiply power by 204 (that is for a 12V reading at AN8)

	clrf		BARGB1		; ms byte
	clrf		AARGB1
	movlw	D'204'	
	movwf	AARGB0		; ls byte
	movf	POWER,w		; power value required
	movwf	BARGB0
	call		FXM88U			; multiply 8 x 8 bits
	
	movf	TEMPD1,w		; divide by AN8 reading
	movwf	BARGB0
; divide
	call		DIV16_8 		; divide 16bits  / 8bits
; 8-bit value in AARGB1
	movf	AARGB1,w
	movwf	PWR_COMP		; compensated power value
	return


; 8 x 8 multiply
FXM88U
      
                MOVLW   H'08'
                MOVWF   LOOPCOUNT
                MOVF    AARGB0,W

LOOPUM0808A
                RRF     BARGB0, F
                BTFSC   STATUS,C
                GOTO    LUM0808NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM0808A

                CLRF    AARGB0
                RETLW   H'00'

LUM0808NAP
                BCF     STATUS,C
                GOTO    LUM0808NA

LOOPUM0808
                RRF     BARGB0, F
                BTFSC   STATUS,C
                ADDWF   AARGB0, F
LUM0808NA      
		RRF    	AARGB0, F
                RRF    	AARGB1, F
                DECFSZ  LOOPCOUNT,F
                GOTO    LOOPUM0808

                return  

;       16/8 Bit Unsigned Fixed Point Divide 16/8 

;       Input:  16 bit unsigned fixed point dividend in AARGB0, AARGB1
;               8 bit unsigned fixed point divisor in BARGB0

;      ;       Output: 16 bit unsigned fixed point quotient in AARGB0, AARGB1
;               8 bit unsigned fixed point remainder in REMB0

;       Result: AARG, REM  <--  AARG / BARG

DIV16_8 
		CLRF            REMB0
                MOVLW           H'08'
                MOVWF           LOOPCOUNT

LOOPU1608A     
		RLF             AARGB0,W
                RLF             REMB0, F
                MOVF            BARGB0,W
                SUBWF           REMB0, F

                BTFSC           STATUS,C
                GOTO            UOK68A          
                ADDWF           REMB0, F
                BCF             STATUS,C
UOK68A  
		RLF             AARGB0, F

                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU1608A

                CLRF            TEMP1

                MOVLW           H'08'
                MOVWF           LOOPCOUNT

LOOPU1608B   
		RLF             AARGB1,W
                RLF             REMB0, F
                RLF             TEMP1, F
                MOVF            BARGB0,W
                SUBWF           REMB0, F
                CLRF            TEMP2
                CLRW
                BTFSS           STATUS,C
                INCFSZ          TEMP2,W
                SUBWF           TEMP1, F

                BTFSC           STATUS,C
                GOTO            UOK68B          
                MOVF            BARGB0,W
                ADDWF           REMB0, F
                CLRF            TEMP2
                CLRW
                BTFSC           STATUS,C
                INCFSZ          TEMP2,W
                ADDWF           TEMP1, F

                BCF             STATUS,C
UOK68B  
	        RLF             AARGB1, F

                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU1608B
		return


	end

